/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        Arbitrary union variant of PGPKeySet.

        $Id: pgpUnionSet.c,v 1.9 2001/01/25 22:11:13 jeffc Exp $
____________________________________________________________________________*/

#include "pgpConfig.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <ctype.h>

#include "pgpKeyPriv.h"
#include "pgpDebug.h"
#include "pgpErrors.h"
#include "pgpTypes.h"
#include "pgpMem.h"
#include "pgpContext.h"


/* Private data for the arbitrary union variant of PGPKeySet */
typedef struct UnionSetPriv
{
	PGPKeySetRef	set1;
	PGPKeySetRef	set2;
} UnionSetPriv;


/*
 * Check to see if a keydbobj is a member of an enumerated subset
 */
	static PGPBoolean
IsMemberOfUnionSet(
	PGPKeySetRef		set,
	PGPKeyDBObjRef		obj)
{
	UnionSetPriv *		priv = (UnionSetPriv *)set->priv;

	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaAddrValid(priv, UnionSetPriv),
		pgpaAddrValid(obj, char)));

	return (priv->set1->isMember(priv->set1, obj)
		 || priv->set2->isMember(priv->set2, obj));
}

/*
 * Check to see if union set is empty
 */
	static PGPBoolean
IsEmptyUnionSet(
	PGPKeySetRef		set )
{
	UnionSetPriv *		priv = (UnionSetPriv *)set->priv;

	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaAddrValid(priv, UnionSetPriv)));

	return (priv->set1->isEmpty(priv->set1)
		 && priv->set2->isEmpty(priv->set2));
}

	static void
DestroyUnionSet(
	PGPKeySetRef		set)
{
	UnionSetPriv *		priv = (UnionSetPriv *)set->priv;

	if( IsntNull( priv ) )
	{
		pgpa((
			pgpaAddrValid(set, PGPKeySet),
			pgpaAddrValid(priv, UnionSetPriv)));

		PGPFreeKeySet(priv->set1);
		PGPFreeKeySet(priv->set2);
		PGPFreeData(priv);
	}
}

	PGPError
pgpGenericUnionOfKeySets(
	PGPKeySetRef		set1,
	PGPKeySetRef		set2,
	PGPKeySetRef *		newSet)
{
	PGPContextRef		context	= PGPPeekKeySetContext(set1);
	UnionSetPriv *		priv;
	PGPKeySetRef		set;
	PGPError			err;

	*newSet = NULL;		/* In case there's an error */

	priv = (UnionSetPriv *)pgpContextMemAlloc(context,
								sizeof(*priv), kPGPMemoryMgrFlags_Clear);
	if (IsNull(priv))
		return kPGPError_OutOfMemory;

	priv->set1 = set1;
	priv->set2 = set2;

	err = pgpNewKeySetInternal(set1->keyDB, &set);
	if (IsPGPError(err))
		return err;
	
	pgpAssertAddrValid(set, PGPKeySet);

	set->priv = priv;
	set->isMember = IsMemberOfUnionSet;
	set->isEmpty = IsEmptyUnionSet;
	set->makeUnion = NULL;
	set->destroy = DestroyUnionSet;

	PGPIncKeySetRefCount(set1);
	PGPIncKeySetRefCount(set2);

	*newSet = set;
	return kPGPError_NoErr;
}

/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
